##############################################################################
##
## Load-Zen-Datatools.ps1 [install|remove}
##
## by Actian Corporation (http://www.actian.com/zen)
##
##############################################################################

    <#
    .NOTES
        Copyright (c) Actian Corporation.  All rights reserved.

        Use of this script is subject to the terms of the Actian Zen license
        agreement under which you licensed the Zen product.  If you did not
        accept the terms of the license agreement, you are not authorized to
        use this script. For the terms of the license, please see the license
        agreement between you and Actian Corporation.
        THE SCRIPT IS PROVIDED "AS IS", WITH NO WARRANTIES.

    .SYNOPSIS
        Integrates the Data Tools for the PSQL ADO.NET Provider v4.5 into
        Visual Studio 2017 or later. Also removes the integration from Visual
        Studio 2017 or later.

    .DESCRIPTION
        Locates the latest installed version of Visual Studio (after VS 2017)
        using the vswhere.exe utility then either installs or removes the
        appropriate Visual Studio Extension package for the Data Tools for PSQL
        ADO.NET Provider v4.5. If Visual Studio is configured for Entity Framework
        development, also copies or deletes the Entity Framework Model First
        templates for the PSQL ADO.NET Provider v4.5 to/from the appropriate
        Visual Studio extensions folder.

    .PARAMETER install
        Uses the VSIXInstaller.exe utility to install the appropriate PSQL ADO.NET
        Provider Data Tools VSIX package. Also copies the Entity Framework Model
        First templates to the appropriate Visual Studio 2017 or later installation
        location if Visual Studio is configured for Entity Framework development.
        A log file for the VSIXInstaller utility output is created in the %TEMP%
        folder with the name "zen.datatools.vsix.install.log".

    .PARAMETER remove
        Uses the VSIXInstaller.exe utility to uninstall the PSQL ADO.NET Provider
        Data Tools VSIX package. Also deletes the Entity Framework Model First
        templates from the Visual Studio 2017 or later installation location if
        Visual Studio is configured for Entity Framework development. A log file
        for the VSIXInstaller utility output is created in the %TEMP% folder with
        the name "zen.datatools.vsix.uninstall.log".

    .EXAMPLE
        PS C:\> .Load-Zen-Datatools.ps1 install
            Installs the VSIX package and copies the Entity Framework Model First
            templates to Visual Studio 2017 or later.

    .EXAMPLE
        PS C:\> .\Load-Zen-Datatools.ps1 remove
            Removes the installed VSIX package and deletes the Entity Framework Model
            First templates from Visual Studio 2017 or later.

    .INPUTS
        None. You cannot pipe objects to this cmdlet.

    .OUTPUTS
        0 : The script completed with no errors (may contain warnings).
        -1: The script returned one or more errors.

    #>

#Requires -version 3.0

param (
    [Parameter(
        Mandatory=$true,
        Position = 0)]
    [ValidateSet('install','remove')]
    [String]
    $InstallMode
)

Set-StrictMode -Version Latest

Import-LocalizedData msg -FileName "Load-Zen-DataTools.psd1" -ErrorAction Ignore

## Check for administrator rights without using the -RunAsAdministrator parameter for the #Requires
## directive (which requires PowerShell 4 or later).
$CurrentUser = New-Object -TypeName Security.Principal.WindowsPrincipal -ArgumentList $([Security.Principal.WindowsIdentity]::GetCurrent())
if ( ! $CurrentUser.IsInRole([Security.Principal.WindowsBuiltinRole]::Administrator)) {Throw $msg.errMustBeAdmin}

## Check PowerShell process is 64-bit on 64-bit OS.
$is32on64=$env:PROCESSOR_ARCHITEW6432 # Only defined if running 32-bit PS on 64-bit Windows.
if ($is32on64) { Throw $msg.errMustBePS64onWin64 }

## Global variable - exists as long as PoSh session is running.
$global:ZenInfo=""

## Script-level variables definitions.
$scriptName = & {$myInvocation.ScriptName}
$scriptPath = & {Split-Path $myInvocation.ScriptName}
$registryPath = "HKLM:\SOFTWARE\Actian\Zen"
$supportedZenMajorVersionLevel = "15"
# Note the vsixPkgName* and vsixId* vars must match the associated VS version checks in GetVSInfo{}.
$vsixPkgName1 = Join-Path $scriptPath 'Zen.VisualStudio.DataTools_2017_2019.vsix'
$vsixId1='PSQLDataTools.d115023a-8951-4aa5-ae36-94d7b1e6f6ad'  # Zen.VisualStudio.DataTools_2017_2019.vsix
$vsixPkgName2 = Join-Path $scriptPath 'Zen.VisualStudio.DataTools_2022.vsix'
$vsixId2='PSQLDataTools.d115023a-8951-4aa5-ae36-94d7b1e6f6ae' # Zen.VisualStudio.DataTools_2022.vsix
# --
$modelFirstIncNames='GenerateZen.Utility*.ttinclude' #ModelFirst Include Template Names
$modelFirstT4Names='SSDLtoZen*.tt' # ModelFirst T4 Template Names
$modelFirstIncludes = join-path $scriptPath "Templates\ModelFirstTemplates\$modelFirstIncNames"
$modelFirstT4 = join-path $scriptPath "Templates\ModelFirstTemplates\$modelFirstT4Names"
$vswhere=join-path $scriptPath 'util\vswhere.exe'
$relPathVSIXInstaller='Common7\IDE\VSIXInstaller.exe'
$relPathEntityFrameworkTools='Common7\IDE\Extensions\Microsoft\Entity Framework Tools'
$relPathEntityFrameworkIncludes=join-path $relPathEntityFrameworkTools 'Templates\Includes'
$relPathEntityFrameworkT4=join-path $relPathEntityFrameworkTools 'DBGen'

function Main {
    ##################################################################################################
    ## Invoked as entry-point function at end of script and defines the order the other defined
    ## functions are called.
    ##################################################################################################
    if ($InstallMode -eq "install") {
        FindProductType
        CheckZenVersion
    }
    GetVSInfo
    RunVsixInstaller
    CopyOrRemoveTemplates
}

function FindProductType {
    # NOTE: This function is duplicated exactly as-is in the Zen Edge WinUWP uninstall script Uninstall-Zen.ps1
    # Any changes here should be kept in-sync with the other script (starting below this line).
    ##########################
    # Sets the installed product type based on one the 'HKLM:\SOFTWARE\Actian\Zen\InstallInfo'
    # sub-keys for Client, Server, Workgroup.
    #
    # Variables set:
    #   [string] $script:InstalledProductType = client|server|reporting
    #   [object] $global:ZenInfo = object methods match InstallInfo registry values.
    #                   $ZenInfo.InstallData
    #                   $ZenInfo.InstallDir64
    #                   $ZenInfo.InstallDir
    #                   $ZenInfo.IsBtrieve
    #                   $ZenInfo.IsReport
    #                   $ZenInfo.IsVx
    #                   $ZenInfo.ProductVersion
    #                   $ZenInfo.VersionLevel
    #                   $ZenInfo.InstallData
    #
    # Note: Must be called before logging is started.
    ##########################
    $script:InstalledProductType=""
    if (test-path $registryPath"\InstallInfo\") {
        $prodName = Get-ChildItem -Name $registryPath"\InstallInfo\"
        Get-ChildItem $registryPath"\InstallInfo\" | ForEach-Object -Process {
        Write-Host "Name: "  $prodName
            switch ($prodName) {
                'Client' {
                    $global:ZenInfo=Get-ItemProperty $registryPath"\InstallInfo\$prodName"
                    $script:InstalledProductType="client"
                    break}
                'Server' {
                    $global:ZenInfo=Get-ItemProperty $registryPath"\InstallInfo\$prodName"
                    if ($ZenInfo.IsVx -eq 0) {
                        $script:InstalledProductType="server"
                    } else {
                        $script:InstalledProductType="vx"
                    }
                    break}
                'Workgroup' {
                    $global:ZenInfo=Get-ItemProperty $registryPath"\InstallInfo\$_"
                    # IsReport is undefined for WG and PoSh throws error if object member not found.
                    $script:InstalledProductType="workgroup" #default
                    Try {
                        if ($global:ZenInfo.IsReport -eq 1) {
                            $script:InstalledProductType="reporting"
                        } 
                    } Catch {}
                    break}
            }
        }
    }
}

function CheckZenVersion {
    ##################################################################################################
    ## Verifies the major version specified in $ZenInfo.ProductVersion (global object set by previous
    ## call to FindProductType) starts with the major version number defined for by script variable
    ## $supportedZenMajorVersionLevel. The script is aborted if the detected major version is not
    ## the same as the defined major version.
    ##################################################################################################
    if ($ZenInfo -ne "") {
        $msg.infoZenInfo -f $ZenInfo.ProductVersion| Write-Output
        $currentVersion=$ZenInfo.ProductVersion[0]+$ZenInfo.ProductVersion[1]
        if ("$supportedZenMajorVersionLevel" -gt $currentVersion ) {
           $msg.errWrongZenVer -f $supportedZenMajorVersionLevel | Write-Error;exit -1
        }
    } else {
        $msg.errNoZenFound | Write-Error;exit -1
    }
}

function GetVSInfo {
    ##################################################################################################
    ## Creates the script level PoSh object "vsInfo" with the member properties specified below. If
    ## Visual Studio 2017 or later is not detected the script displays an error message and aborts.
    ## If the retrieved path to the VSIXInstaller utility is invalid the script also displays an error
    ## message and aborts.
    ##
    ## object member properties:
    ##    vsInfo.installDir    : value of property 'installationPath' returned by vswhere utility.
    ##    vsInfo.model1_inc    : path to Entity Framework Tools\Templates\Includes folder if it exists,
    ##                           empty string if it does not exist.
    ##    vsInfo.model1_t4     : path to Entity Framework Tools\DBGen folder if it exists, empty string
    ##                           if it does not exist.
    ##    vsInfo.version       : value of property 'installationVersion' returned by vswhere utility.
    ##    vsInfo.vsixInstaller : path to the VSIXInstaller utility (must exist).
    ##################################################################################################
    $vsInstallDirFound=$false

    ## Get installed version and location of VS. See the following URL for information on valid version
    ## specifiers for VS: https://docs.microsoft.com/en-us/visualstudio/extensibility/vsix-extension-schema-2-0-reference

    ## Examples to obtain specific version ranges.
    # Visual Studio 2022: version "[17.0,18.0)"
    # $vsVersion=& $vswhere -nologo -products * -property installationVersion -version [17.0,18.0)

    # Visual Studio 2019: version "[16.0,17.0)"
    # $vsVersion=& $vswhere -nologo -products * -property installationVersion -version [16.0,17.0)

    # Visual Studio 2017: version "[15.0,16.0)"
    # $vsVersion=& $vswhere -nologo -products * -property installationVersion -version [15.0,16.0)

    # Just get the latest installed version.  This matches VS 2017 and later because VS 2015 and
    # earlier require the "-legacy" option to retrieve the installationVersion property so without
    # that option specified VS 2017 is the earliest that would be detected even if VS 2015 or earlier
    # is also installed.
    $vsVersion=& $vswhere -nologo -latest -products * -property installationVersion
    if ($vsVersion) {
        ## VS 2017 or later.
        $vsInstallDir=& $vswhere -nologo -latest -products * -property installationPath
        if ($vsInstallDir -and (Test-Path -Path $vsInstallDir)) {$vsInstallDirFound=$True}
    }

    ## Abort script if $vsVersion is not "15.0" or later (Visual Studio 2017 or later) or if the
    ## installation path does not exist.
    if ( (!$vsVersion) -or ($vsVersion -lt "15.0") -or (!$vsInstallDirFound) ) {$msg.errNoVSFound -f "2017"| Write-Error;exit -1}

    ## Verify the location of the VSIXInstaller utility required to install/remove the VSIX package.
    $vsixInstaller=Join-Path $vsInstallDir $relPathVSIXInstaller
    if (! (test-path $vsixInstaller)) {$msg.errNoVSIXInstaller | Write-Error;exit -1}

    ## Verify the location of the Entity Framework Tools "Templates\Includes" and "DBGen" folders
    ## (sets corresponding vsInfo properties to empty string if the folder doesn't exist).
    $entityFrameworkIncludesPath=Join-Path $vsInstallDir $relPathEntityFrameworkIncludes
    if (!(test-path $entityFrameworkIncludesPath)) {$entityFrameworkIncludesPath=""}
    $entityFrameworkDBGenPath=Join-Path $vsInstallDir $relPathEntityFrameworkT4
    if (!(test-path $entityFrameworkDBGenPath)) {$entityFrameworkDBGenPath=""}

    ## If the script has gotten this far VS 2017 or later is installed.  We have different VSIX
    ## packages for different versions of VS so we need to know which one to use.
    if ( ($vsVersion -lt "17.0" ) ) {
        ## VS 2017 or VS 2019
        $vsixPkgToUse=$vsixPkgName1
        $vsixIdToUse=$vsixId1
    } else {
        ## VS 2022 or later
        $vsixPkgToUse=$vsixPkgName2
        $vsixIdToUse=$vsixId2
    }

    ## Create PoSh object vsInfo.
    $script:vsInfo = @{
        'installDir'=$vsInstallDir;
        'model1_inc'=$entityFrameworkIncludesPath;
        'model1_t4'=$entityFrameworkDBGenPath;
        'version'=$vsVersion;
        'vsixInstaller'=$vsixInstaller;
        'vsixPackage'=$vsixPkgToUse;
        'vsixID'=$vsixIdToUse;
    }

    ## Output object contents.
    $script:vsInfo | Write-Output

    ## Output VS version and location info.
    $msg.infoVisualStudioInfo -f $vsInfo.version,$vsInfo.installDir | Write-Output
}

function RunVsixInstaller {
    ##################################################################################################
    ## Runs VSIXInstaller utility to install or remove the appropriate VSIX package.  If the
    ## VSIXInstaller.exe process cannot be started an error message is displayed and the script is
    ## aborted, otherwise an informational message instructing the user to check the VSIXInstaller log
    ## file is displayed.  The VSIX package name and package ID are stored in the vsInfo object
    ## created by GetVSInfo{}.
    ##################################################################################################

    ## Set the VSIX package name and install/uninstall options based on Visual Studio version.
    $vsixInstallArgs="/quiet","/logFile:Zen.DataTools.VSIX.install.log",$vsInfo.vsixPackage
    $vsixInstallMsg=$msg.infoInstalledVsixPkgLatestWithLog -f [string]"Zen.DataTools.VSIX.install.log"
    $vsixInstallErr=$msg.errVsixInstallWithLog -f [string]" (Zen.DataTools.VSIX.install.log)"
    $vsixRemovalArgs="/quiet","/logFile:Zen.DataTools.VSIX.uninstall.log",("/uninstall:"+$vsInfo.vsixID)
    $vsixRemovalMsg=$msg.infoRemovedVsixPkgLatestWithLog -f [string]"Zen.DataTools.VSIX.uninstall.log"
    $vsixRemovalErr=$msg.errVsixUninstallWithLog -f [string]" (Zen.DataTools.VSIX.uninstall.log)"


    ## Install or remove the VSIX package based on $InstallMode.
    if ($InstallMode -eq "install") {
        Try {
           $msg.infoInstallingVsixPkg | Write-Output
           Start-Process -FilePath $vsInfo.vsixInstaller -ArgumentList $vsixInstallArgs -NoNewWindow -Wait
           $vsixInstallMsg | Write-Output
        }
        Catch {$vsixInstallErr | Write-Error;exit -1}
    } else {
        Try {
            $msg.infoRemovingVsixPkg | Write-Output
            Start-Process -FilePath $vsInfo.vsixInstaller -ArgumentList $vsixRemovalArgs -NoNewWindow -Wait
            $vsixRemovalMsg | Write-Output
        }
        Catch {$vsixRemovalErr | Write-Error;exit -1}
    }
}

function CopyOrRemoveTemplates {
    ##################################################################################################
    ## Copies or removes the Entity Framework Model First templates to/from the appropriate Visual
    ## Studio folder.  If the path to the Entity Framework Tools folder cannot be located this step
    ## is skipped.
    ##################################################################################################

    if ($InstallMode -eq "install") {  ## Copy templates.
        Try {
            ## Verify Entity Framework Tools folder and copy templates.
            $entityFrameworkTools=join-path $vsInfo.installDir $relPathEntityFrameworkTools
            if (Test-Path $entityFrameworkTools) {

                ## Model First include templates.
                Copy-Item -Path $modelFirstIncludes -Destination $vsInfo.model1_inc -Force
                if ($?) {$msg.infoAddedModelFirstIncludeTemplates | Write-Output}

                ## Model First T4 templates.
                Copy-Item -Path $modelFirstT4 -Destination $vsInfo.model1_t4 -Force
                if ($?) {$msg.infoAddedModelFirstT4Templates | Write-Output}

            }
        }
        Catch {$msg.warnNoTemplates -f [string]"$relPathEntityFrameworkTools" | Write-Warning}
    } else {  ## Remove templates.
        Try {
            ## Verify Model First include templates exist and remove them.
            $removeIncludes=join-path $vsInfo.model1_inc $modelFirstIncNames
            if (Test-Path $removeIncludes) {
                remove-item -Path $removeIncludes -Force
                if ($?) {$msg.infoRemovedModelFirstIncludeTemplates | Write-Output}
            }

            ## Verify Model First T4 templates exist and remove them.
            $removeT4=join-path $vsInfo.model1_t4 $modelFirstT4Names
            if (Test-Path $removeT4) {
                remove-item -Path $removeT4 -Force
                if ($?) {$msg.infoRemovedModelFirstT4Templates | Write-Output}
            }
        }
        Catch {$msg.warnNoTemplates -f [string]"$relPathEntityFrameworkTools" | Write-Warning}
    }
}

## Invoke Main function.
"" | Write-Output
$msg.infoRunning -f $MyInvocation.InvocationName, $InstallMode | Write-Output
. Main
$msg.infoEnd -f $MyInvocation.InvocationName, $InstallMode | Write-Output
"" | Write-Output
exit 0
# SIG # Begin signature block
# MIIpUgYJKoZIhvcNAQcCoIIpQzCCKT8CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCBrZBzmb7TugUXB
# M2jFqPGm6XOsyd7RQP9hVyIPKQ7jM6CCDiowggawMIIEmKADAgECAhAIrUCyYNKc
# TJ9ezam9k67ZMA0GCSqGSIb3DQEBDAUAMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQK
# EwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNV
# BAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0yMTA0MjkwMDAwMDBaFw0z
# NjA0MjgyMzU5NTlaMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAw
# ggIKAoICAQDVtC9C0CiteLdd1TlZG7GIQvUzjOs9gZdwxbvEhSYwn6SOaNhc9es0
# JAfhS0/TeEP0F9ce2vnS1WcaUk8OoVf8iJnBkcyBAz5NcCRks43iCH00fUyAVxJr
# Q5qZ8sU7H/Lvy0daE6ZMswEgJfMQ04uy+wjwiuCdCcBlp/qYgEk1hz1RGeiQIXhF
# LqGfLOEYwhrMxe6TSXBCMo/7xuoc82VokaJNTIIRSFJo3hC9FFdd6BgTZcV/sk+F
# LEikVoQ11vkunKoAFdE3/hoGlMJ8yOobMubKwvSnowMOdKWvObarYBLj6Na59zHh
# 3K3kGKDYwSNHR7OhD26jq22YBoMbt2pnLdK9RBqSEIGPsDsJ18ebMlrC/2pgVItJ
# wZPt4bRc4G/rJvmM1bL5OBDm6s6R9b7T+2+TYTRcvJNFKIM2KmYoX7BzzosmJQay
# g9Rc9hUZTO1i4F4z8ujo7AqnsAMrkbI2eb73rQgedaZlzLvjSFDzd5Ea/ttQokbI
# YViY9XwCFjyDKK05huzUtw1T0PhH5nUwjewwk3YUpltLXXRhTT8SkXbev1jLchAp
# QfDVxW0mdmgRQRNYmtwmKwH0iU1Z23jPgUo+QEdfyYFQc4UQIyFZYIpkVMHMIRro
# OBl8ZhzNeDhFMJlP/2NPTLuqDQhTQXxYPUez+rbsjDIJAsxsPAxWEQIDAQABo4IB
# WTCCAVUwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQUaDfg67Y7+F8Rhvv+
# YXsIiGX0TkIwHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4cD08wDgYDVR0P
# AQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHcGCCsGAQUFBwEBBGswaTAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsGAQUFBzAC
# hjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9v
# dEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2VydC5j
# b20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAcBgNVHSAEFTATMAcGBWeBDAED
# MAgGBmeBDAEEATANBgkqhkiG9w0BAQwFAAOCAgEAOiNEPY0Idu6PvDqZ01bgAhql
# +Eg08yy25nRm95RysQDKr2wwJxMSnpBEn0v9nqN8JtU3vDpdSG2V1T9J9Ce7FoFF
# UP2cvbaF4HZ+N3HLIvdaqpDP9ZNq4+sg0dVQeYiaiorBtr2hSBh+3NiAGhEZGM1h
# mYFW9snjdufE5BtfQ/g+lP92OT2e1JnPSt0o618moZVYSNUa/tcnP/2Q0XaG3Ryw
# YFzzDaju4ImhvTnhOE7abrs2nfvlIVNaw8rpavGiPttDuDPITzgUkpn13c5Ubdld
# AhQfQDN8A+KVssIhdXNSy0bYxDQcoqVLjc1vdjcshT8azibpGL6QB7BDf5WIIIJw
# 8MzK7/0pNVwfiThV9zeKiwmhywvpMRr/LhlcOXHhvpynCgbWJme3kuZOX956rEnP
# LqR0kq3bPKSchh/jwVYbKyP/j7XqiHtwa+aguv06P0WmxOgWkVKLQcBIhEuWTatE
# QOON8BUozu3xGFYHKi8QxAwIZDwzj64ojDzLj4gLDb879M4ee47vtevLt/B3E+bn
# KD+sEq6lLyJsQfmCXBVmzGwOysWGw/YmMwwHS6DTBwJqakAwSEs0qFEgu60bhQji
# WQ1tygVQK+pKHJ6l/aCnHwZ05/LWUpD9r4VIIflXO7ScA+2GRfS0YW6/aOImYIbq
# yK+p/pQd52MbOoZWeE4wggdyMIIFWqADAgECAhALug4gRzX+ySpZsBzdR9OMMA0G
# CSqGSIb3DQEBCwUAMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwHhcNMjMwNDI0MDAwMDAwWhcNMjYwNDIz
# MjM1OTU5WjB6MQswCQYDVQQGEwJVUzEOMAwGA1UECBMFVGV4YXMxEzARBgNVBAcT
# ClJvdW5kIFJvY2sxGzAZBgNVBAoTEkFjdGlhbiBDb3Jwb3JhdGlvbjEMMAoGA1UE
# CxMDWmVuMRswGQYDVQQDExJBY3RpYW4gQ29ycG9yYXRpb24wggIiMA0GCSqGSIb3
# DQEBAQUAA4ICDwAwggIKAoICAQCmrbJewixh8zYkjt5G34WYugoGnAfUHCzJmXGn
# sNqZ7jSNZR8Fy3Z9aFFUkwYULY2CQxNNC/GSdnt1lB0TM1ggSido3ttaJ2FaUD9F
# oi1keyc9pR8cKjtt8R1go+BNHy4MIPrr6HbI9ObXxGwP5WVAW72jaOW2rJcGOKuB
# wG642zwhAA4wJxRJKV394PxuUsLmqqyarb8wdvAhZJUP0OZiV5ERJ0ZglBv4qdPt
# e0u4pZCpbhhlPEptFHX2bEdY9eKAeqZClenQyzJqJkOgpeGPjoN8mh1FLeLlwSjs
# AdiAL4KBHyAQb2kGEU7uJaKpyoq1VxSerNWtZUmihQMP9EJg7cHSZPJNj2HBhhOR
# r1NbEzyZjx2I1G/Ib69A0R6CaiEXyJnLJxg4R5VazeM87YS23ML0jFyDiCzfnQp5
# W+ls8i0MHhqqVqpsRZPViHHjJtNpcAR7KsbRrlsXL9c5kMqNvKxDvZeV0oVSECAd
# 3Djtr0pfSvXU0eKIl8i3LDimc+VcTtbbfs93TGDyVaNb9uDTlAKhk46PaYhamNqr
# ZCkWWiNSdnglito6wKaOa91AnBN2WbQxOEEpz8UEvNPYM/+HGYIwefXspJz4wvAB
# yB9AYShQ4oUGPnQ/+fPWZFZ/yTi7Gpb69FedVltEiL7A+3Ds7UKZ/269YhTQYSGh
# P5R8HQIDAQABo4ICAzCCAf8wHwYDVR0jBBgwFoAUaDfg67Y7+F8Rhvv+YXsIiGX0
# TkIwHQYDVR0OBBYEFLZN9fT8GcYstZ/eHlOoSwqtoRPtMA4GA1UdDwEB/wQEAwIH
# gDATBgNVHSUEDDAKBggrBgEFBQcDAzCBtQYDVR0fBIGtMIGqMFOgUaBPhk1odHRw
# Oi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRDb2RlU2lnbmlu
# Z1JTQTQwOTZTSEEzODQyMDIxQ0ExLmNybDBToFGgT4ZNaHR0cDovL2NybDQuZGln
# aWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0Q29kZVNpZ25pbmdSU0E0MDk2U0hB
# Mzg0MjAyMUNBMS5jcmwwPgYDVR0gBDcwNTAzBgZngQwBBAEwKTAnBggrBgEFBQcC
# ARYbaHR0cDovL3d3dy5kaWdpY2VydC5jb20vQ1BTMIGUBggrBgEFBQcBAQSBhzCB
# hDAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMFwGCCsGAQUF
# BzAChlBodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVk
# RzRDb2RlU2lnbmluZ1JTQTQwOTZTSEEzODQyMDIxQ0ExLmNydDAJBgNVHRMEAjAA
# MA0GCSqGSIb3DQEBCwUAA4ICAQCWGybROBuJGQ2OT6P6pJtQptFx2fprQP+Ttb5L
# 97xCPKnI72HljOJUKIiC+QJCqolKBFvtxbtzsFap4fXkBQVsDcrNijMLjX5IxAFv
# l2N7kQ03POGihblkie2n+HgX0Frnd9PrJxSBalzylmsNtHCvhgwdnSZzPBb9pGW8
# zLJjsEiFGOPvH6UP2TfLNZHpys10DpM5U44mw9h034EHuNafF4Dbxp7heu58jtWS
# QDzFmVe2+4/tk5eHAHx0IuhAWYhtTlygMZK54oOp0DHSRnQKMfeqa7XUS5pwVFR1
# MLbZETlr8SS2Y5AnPHKGE4vFOVXOznwEmYUTj/+bRnL0cm7ndeLsAfHVQsJjmF4K
# gcNaqMaAuAV80M4LrFt+bV0JUC83cu2RmrMKQyO1FEm7yp7VcQniVIbwBv1w+rOh
# aKAW+6Zoj4n6QrpYpdIdjs5gY2crwzUs5wst16bkVHQh+vFX6KCax41RcXTvLTbI
# T4Hnz3zSOTQKLHunfQs4/QyUd8LoAPnKJGH9Dwgs3BESIbFcdLDuzQ9Gza9dUaBX
# eswMXJug5CaO8gblx99JD6jZBGV6hUb9+72UTKWy+3Yv2KkFPmPLxUqyrAoG+MR9
# DOEeMZP10Qjfu7OeRZoR6oSeWLQJUUS2zybxFq4Xxx6C5v58ZUtO9gOjLhrwB72m
# mPuBWDGCGn4wghp6AgEBMH0waTELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lD
# ZXJ0LCBJbmMuMUEwPwYDVQQDEzhEaWdpQ2VydCBUcnVzdGVkIEc0IENvZGUgU2ln
# bmluZyBSU0E0MDk2IFNIQTM4NCAyMDIxIENBMQIQC7oOIEc1/skqWbAc3UfTjDAN
# BglghkgBZQMEAgEFAKCBljAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAqBgorBgEEAYI3AgEMMRwwGqEYgBZo
# dHRwOlxcd3d3LmFjdGlhbi5jb20gMC8GCSqGSIb3DQEJBDEiBCBP4x0paaajMZYC
# hXTgO1QspIttar7aCRG82Nzi7rhQrzANBgkqhkiG9w0BAQEFAASCAgAOZF7477gk
# XGq7uQOWYerU7vc9G4PgNqD/prYfI6LD/CJmQxmdTBW7tLIszBcka900EjtO2j4v
# 2P8rs8TxO4tQESBtd3Gk1LvdtPaWepQ1yxVv+akhlTT3so2MQJxHzJFh8scJ+WYR
# BQeSynuvMnNMsaUzNIp/9oTmp2aoQudkNFnsASxnARnCdzNn3iEWTeeo0R0KNXGc
# PG3OCjGdnQEUoTfOCkaJ/QTsRyruTHMUN2pIUfau+CkwJzWF+dDM1pdACjCMfIyN
# Q2IO4zs78Eoy5FLgJE6qsPNb0xQPv5+5t4DKHqZcSxhwKzJhiUzbCSTNOS8+wbHs
# V367TDYSbdfHZhDiZI9+asXeGK411hQLK6yNVH5EKCMLQz3lR9yP2wWwIbkeEoeW
# IzYHGhdnKE7KygNmykVxJg8f+7YVLYhfD3ybuwn+9gcqGkhQF94fpooAyXHJYsE8
# iIEwZGr7fUVsgph06yofyv0ay4RUXVegwAdLkTZpf9fdDBwRB9jIX8B0QLzvmOYc
# LGW07E5O3oqp7G7xh5BKTB0boQaB94bYX2kCIjv/gDMi6EM1Sml/rbr1bvIaK/uH
# lzmiK1LaX1B56mvELdvRHkFY+7iQ6UBPQ/QdY6SJxc0p/In9v20UhuImqDj8dpIj
# 69FKhLfWNl+pn35ooc7+3Y8F3+gGoJhKxqGCFzkwghc1BgorBgEEAYI3AwMBMYIX
# JTCCFyEGCSqGSIb3DQEHAqCCFxIwghcOAgEDMQ8wDQYJYIZIAWUDBAIBBQAwdwYL
# KoZIhvcNAQkQAQSgaARmMGQCAQEGCWCGSAGG/WwHATAxMA0GCWCGSAFlAwQCAQUA
# BCABY+LaLJRf+WrimuebAhcsUZy0XNAigL2JOux1scz26gIQElvdgUzVYzXIUxo/
# zVGmDRgPMjAyNTA2MjcwMTA5MDhaoIITAzCCBrwwggSkoAMCAQICEAuuZrxaun+V
# h8b56QTjMwQwDQYJKoZIhvcNAQELBQAwYzELMAkGA1UEBhMCVVMxFzAVBgNVBAoT
# DkRpZ2lDZXJ0LCBJbmMuMTswOQYDVQQDEzJEaWdpQ2VydCBUcnVzdGVkIEc0IFJT
# QTQwOTYgU0hBMjU2IFRpbWVTdGFtcGluZyBDQTAeFw0yNDA5MjYwMDAwMDBaFw0z
# NTExMjUyMzU5NTlaMEIxCzAJBgNVBAYTAlVTMREwDwYDVQQKEwhEaWdpQ2VydDEg
# MB4GA1UEAxMXRGlnaUNlcnQgVGltZXN0YW1wIDIwMjQwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQC+anOf9pUhq5Ywultt5lmjtej9kR8YxIg7apnjpcH9
# CjAgQxK+CMR0Rne/i+utMeV5bUlYYSuuM4vQngvQepVHVzNLO9RDnEXvPghCaft0
# djvKKO+hDu6ObS7rJcXa/UKvNminKQPTv/1+kBPgHGlP28mgmoCw/xi6FG9+Un1h
# 4eN6zh926SxMe6We2r1Z6VFZj75MU/HNmtsgtFjKfITLutLWUdAoWle+jYZ49+wx
# GE1/UXjWfISDmHuI5e/6+NfQrxGFSKx+rDdNMsePW6FLrphfYtk/FLihp/feun0e
# V+pIF496OVh4R1TvjQYpAztJpVIfdNsEvxHofBf1BWkadc+Up0Th8EifkEEWdX4r
# A/FE1Q0rqViTbLVZIqi6viEk3RIySho1XyHLIAOJfXG5PEppc3XYeBH7xa6VTZ3r
# OHNeiYnY+V4j1XbJ+Z9dI8ZhqcaDHOoj5KGg4YuiYx3eYm33aebsyF6eD9MF5IDb
# PgjvwmnAalNEeJPvIeoGJXaeBQjIK13SlnzODdLtuThALhGtyconcVuPI8AaiCai
# JnfdzUcb3dWnqUnjXkRFwLtsVAxFvGqsxUA2Jq/WTjbnNjIUzIs3ITVC6VBKAOlb
# 2u29Vwgfta8b2ypi6n2PzP0nVepsFk8nlcuWfyZLzBaZ0MucEdeBiXL+nUOGhCjl
# +QIDAQABo4IBizCCAYcwDgYDVR0PAQH/BAQDAgeAMAwGA1UdEwEB/wQCMAAwFgYD
# VR0lAQH/BAwwCgYIKwYBBQUHAwgwIAYDVR0gBBkwFzAIBgZngQwBBAIwCwYJYIZI
# AYb9bAcBMB8GA1UdIwQYMBaAFLoW2W1NhS9zKXaaL3WMaiCPnshvMB0GA1UdDgQW
# BBSfVywDdw4oFZBmpWNe7k+SH3agWzBaBgNVHR8EUzBRME+gTaBLhklodHRwOi8v
# Y3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRSU0E0MDk2U0hBMjU2
# VGltZVN0YW1waW5nQ0EuY3JsMIGQBggrBgEFBQcBAQSBgzCBgDAkBggrBgEFBQcw
# AYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMFgGCCsGAQUFBzAChkxodHRwOi8v
# Y2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRSU0E0MDk2U0hB
# MjU2VGltZVN0YW1waW5nQ0EuY3J0MA0GCSqGSIb3DQEBCwUAA4ICAQA9rR4fdplb
# 4ziEEkfZQ5H2EdubTggd0ShPz9Pce4FLJl6reNKLkZd5Y/vEIqFWKt4oKcKz7wZm
# Xa5VgW9B76k9NJxUl4JlKwyjUkKhk3aYx7D8vi2mpU1tKlY71AYXB8wTLrQeh83p
# XnWwwsxc1Mt+FWqz57yFq6laICtKjPICYYf/qgxACHTvypGHrC8k1TqCeHk6u4I/
# VBQC9VK7iSpU5wlWjNlHlFFv/M93748YTeoXU/fFa9hWJQkuzG2+B7+bMDvmgF8V
# lJt1qQcl7YFUMYgZU1WM6nyw23vT6QSgwX5Pq2m0xQ2V6FJHu8z4LXe/371k5QrN
# 9FQBhLLISZi2yemW0P8ZZfx4zvSWzVXpAb9k4Hpvpi6bUe8iK6WonUSV6yPlMwer
# wJZP/Gtbu3CKldMnn+LmmRTkTXpFIEB06nXZrDwhCGED+8RsWQSIXZpuG4WLFQOh
# tloDRWGoCwwc6ZpPddOFkM2LlTbMcqFSzm4cd0boGhBq7vkqI1uHRz6Fq1IX7TaR
# QuR+0BGOzISkcqwXu7nMpFu3mgrlgbAW+BzikRVQ3K2YHcGkiKjA4gi4OA/kz1YC
# sdhIBHXqBzR0/Zd2QwQ/l4Gxftt/8wY3grcc/nS//TVkej9nmUYu83BDtccHHXKi
# bMs/yXHhDXNkoPIdynhVAku7aRZOwqw6pDCCBq4wggSWoAMCAQICEAc2N7ckVHzY
# R6z9KGYqXlswDQYJKoZIhvcNAQELBQAwYjELMAkGA1UEBhMCVVMxFTATBgNVBAoT
# DERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEhMB8GA1UE
# AxMYRGlnaUNlcnQgVHJ1c3RlZCBSb290IEc0MB4XDTIyMDMyMzAwMDAwMFoXDTM3
# MDMyMjIzNTk1OVowYzELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJ
# bmMuMTswOQYDVQQDEzJEaWdpQ2VydCBUcnVzdGVkIEc0IFJTQTQwOTYgU0hBMjU2
# IFRpbWVTdGFtcGluZyBDQTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIB
# AMaGNQZJs8E9cklRVcclA8TykTepl1Gh1tKD0Z5Mom2gsMyD+Vr2EaFEFUJfpIjz
# aPp985yJC3+dH54PMx9QEwsmc5Zt+FeoAn39Q7SE2hHxc7Gz7iuAhIoiGN/r2j3E
# F3+rGSs+QtxnjupRPfDWVtTnKC3r07G1decfBmWNlCnT2exp39mQh0YAe9tEQYnc
# fGpXevA3eZ9drMvohGS0UvJ2R/dhgxndX7RUCyFobjchu0CsX7LeSn3O9TkSZ+8O
# pWNs5KbFHc02DVzV5huowWR0QKfAcsW6Th+xtVhNef7Xj3OTrCw54qVI1vCwMROp
# VymWJy71h6aPTnYVVSZwmCZ/oBpHIEPjQ2OAe3VuJyWQmDo4EbP29p7mO1vsgd4i
# FNmCKseSv6De4z6ic/rnH1pslPJSlRErWHRAKKtzQ87fSqEcazjFKfPKqpZzQmif
# tkaznTqj1QPgv/CiPMpC3BhIfxQ0z9JMq++bPf4OuGQq+nUoJEHtQr8FnGZJUlD0
# UfM2SU2LINIsVzV5K6jzRWC8I41Y99xh3pP+OcD5sjClTNfpmEpYPtMDiP6zj9Ne
# S3YSUZPJjAw7W4oiqMEmCPkUEBIDfV8ju2TjY+Cm4T72wnSyPx4JduyrXUZ14mCj
# WAkBKAAOhFTuzuldyF4wEr1GnrXTdrnSDmuZDNIztM2xAgMBAAGjggFdMIIBWTAS
# BgNVHRMBAf8ECDAGAQH/AgEAMB0GA1UdDgQWBBS6FtltTYUvcyl2mi91jGogj57I
# bzAfBgNVHSMEGDAWgBTs1+OC0nFdZEzfLmc/57qYrhwPTzAOBgNVHQ8BAf8EBAMC
# AYYwEwYDVR0lBAwwCgYIKwYBBQUHAwgwdwYIKwYBBQUHAQEEazBpMCQGCCsGAQUF
# BzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wQQYIKwYBBQUHMAKGNWh0dHA6
# Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRSb290RzQuY3J0
# MEMGA1UdHwQ8MDowOKA2oDSGMmh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdp
# Q2VydFRydXN0ZWRSb290RzQuY3JsMCAGA1UdIAQZMBcwCAYGZ4EMAQQCMAsGCWCG
# SAGG/WwHATANBgkqhkiG9w0BAQsFAAOCAgEAfVmOwJO2b5ipRCIBfmbW2CFC4bAY
# LhBNE88wU86/GPvHUF3iSyn7cIoNqilp/GnBzx0H6T5gyNgL5Vxb122H+oQgJTQx
# Z822EpZvxFBMYh0MCIKoFr2pVs8Vc40BIiXOlWk/R3f7cnQU1/+rT4osequFzUNf
# 7WC2qk+RZp4snuCKrOX9jLxkJodskr2dfNBwCnzvqLx1T7pa96kQsl3p/yhUifDV
# inF2ZdrM8HKjI/rAJ4JErpknG6skHibBt94q6/aesXmZgaNWhqsKRcnfxI2g55j7
# +6adcq/Ex8HBanHZxhOACcS2n82HhyS7T6NJuXdmkfFynOlLAlKnN36TU6w7HQhJ
# D5TNOXrd/yVjmScsPT9rp/Fmw0HNT7ZAmyEhQNC3EyTN3B14OuSereU0cZLXJmvk
# OHOrpgFPvT87eK1MrfvElXvtCl8zOYdBeHo46Zzh3SP9HSjTx/no8Zhf+yvYfvJG
# nXUsHicsJttvFXseGYs2uJPU5vIXmVnKcPA3v5gA3yAWTyf7YGcWoWa63VXAOimG
# sJigK+2VQbc61RWYMbRiCQ8KvYHZE/6/pNHzV9m8BPqC3jLfBInwAM1dwvnQI38A
# C+R2AibZ8GV2QqYphwlHK+Z/GqSFD/yYlvZVVCsfgPrA8g4r5db7qS9EFUrnEw4d
# 2zc4GqEr9u3WfPwwggWNMIIEdaADAgECAhAOmxiO+dAt5+/bUOIIQBhaMA0GCSqG
# SIb3DQEBDAUAMGUxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMx
# GTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xJDAiBgNVBAMTG0RpZ2lDZXJ0IEFz
# c3VyZWQgSUQgUm9vdCBDQTAeFw0yMjA4MDEwMDAwMDBaFw0zMTExMDkyMzU5NTla
# MGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsT
# EHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9v
# dCBHNDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAL/mkHNo3rvkXUo8
# MCIwaTPswqclLskhPfKK2FnC4SmnPVirdprNrnsbhA3EMB/zG6Q4FutWxpdtHauy
# efLKEdLkX9YFPFIPUh/GnhWlfr6fqVcWWVVyr2iTcMKyunWZanMylNEQRBAu34Lz
# B4TmdDttceItDBvuINXJIB1jKS3O7F5OyJP4IWGbNOsFxl7sWxq868nPzaw0QF+x
# embud8hIqGZXV59UWI4MK7dPpzDZVu7Ke13jrclPXuU15zHL2pNe3I6PgNq2kZhA
# kHnDeMe2scS1ahg4AxCN2NQ3pC4FfYj1gj4QkXCrVYJBMtfbBHMqbpEBfCFM1Lyu
# GwN1XXhm2ToxRJozQL8I11pJpMLmqaBn3aQnvKFPObURWBf3JFxGj2T3wWmIdph2
# PVldQnaHiZdpekjw4KISG2aadMreSx7nDmOu5tTvkpI6nj3cAORFJYm2mkQZK37A
# lLTSYW3rM9nF30sEAMx9HJXDj/chsrIRt7t/8tWMcCxBYKqxYxhElRp2Yn72gLD7
# 6GSmM9GJB+G9t+ZDpBi4pncB4Q+UDCEdslQpJYls5Q5SUUd0viastkF13nqsX40/
# ybzTQRESW+UQUOsxxcpyFiIJ33xMdT9j7CFfxCBRa2+xq4aLT8LWRV+dIPyhHsXA
# j6KxfgommfXkaS+YHS312amyHeUbAgMBAAGjggE6MIIBNjAPBgNVHRMBAf8EBTAD
# AQH/MB0GA1UdDgQWBBTs1+OC0nFdZEzfLmc/57qYrhwPTzAfBgNVHSMEGDAWgBRF
# 66Kv9JLLgjEtUYunpyGd823IDzAOBgNVHQ8BAf8EBAMCAYYweQYIKwYBBQUHAQEE
# bTBrMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wQwYIKwYB
# BQUHMAKGN2h0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFzc3Vy
# ZWRJRFJvb3RDQS5jcnQwRQYDVR0fBD4wPDA6oDigNoY0aHR0cDovL2NybDMuZGln
# aWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNybDARBgNVHSAECjAI
# MAYGBFUdIAAwDQYJKoZIhvcNAQEMBQADggEBAHCgv0NcVec4X6CjdBs9thbX979X
# B72arKGHLOyFXqkauyL4hxppVCLtpIh3bb0aFPQTSnovLbc47/T/gLn4offyct4k
# vFIDyE7QKt76LVbP+fT3rDB6mouyXtTP0UNEm0Mh65ZyoUi0mcudT6cGAxN3J0TU
# 53/oWajwvy8LpunyNDzs9wPHh6jSTEAZNUZqaVSwuKFWjuyk1T3osdz9HNj0d1pc
# VIxv76FQPfx2CWiEn2/K2yCNNWAcAgPLILCsWKAOQGPFmCLBsln1VWvPJ6tsds5v
# Iy30fnFqI2si/xK4VC0nftg62fC2h5b9W9FcrBjDTZ9ztwGpn1eqXijiuZQxggN2
# MIIDcgIBATB3MGMxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5j
# LjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3RlZCBHNCBSU0E0MDk2IFNIQTI1NiBU
# aW1lU3RhbXBpbmcgQ0ECEAuuZrxaun+Vh8b56QTjMwQwDQYJYIZIAWUDBAIBBQCg
# gdEwGgYJKoZIhvcNAQkDMQ0GCyqGSIb3DQEJEAEEMBwGCSqGSIb3DQEJBTEPFw0y
# NTA2MjcwMTA5MDhaMCsGCyqGSIb3DQEJEAIMMRwwGjAYMBYEFNvThe5i29I+e+T2
# cUhQhyTVhltFMC8GCSqGSIb3DQEJBDEiBCAMEHeJGBeI1tcDcyF4CfgSLnnUtdLx
# iB8yxAlQzU6RlTA3BgsqhkiG9w0BCRACLzEoMCYwJDAiBCB2dp+o8mMvH0MLOiMw
# rtZWdf7Xc9sF1mW5BZOYQ4+a2zANBgkqhkiG9w0BAQEFAASCAgB2sxD8hVhopKvc
# wlR35LxqVT40F3EF9E/K5cY+gNufwnQ/4YSvlsR4T0fnLwbcOrRcKRc7xuK28JUL
# EMOx38fmbZHZ8RECZt1hAhCZkU1/Fp34wc3tOXY7fh9Ad0eIoG0YbyZwLvAJ3jNW
# 9FQRVRjwHV1CjogbPV8IEOokFlm0TjHWqNn9iw/KioohbPeIV1uHhXOGj8sBsCJb
# 9f8jkKy9O1BqEdA0BHruyAY8GCEy+eJFsAj8zRznlkQEEOU1gd9sSpHCOajvdeU9
# +HxyDnozGqcNEFqaB2LhVej/tFhtvZWDBxq6dzBbchXaupjoiBqKq5KQ7t+C8c8U
# RgDYPfxmjGf+6xhUI8Fo7epsroL4bCLSElNPyRKmqF8VAR9qXdUG1QCWcwknGxPX
# fd1J9RUs36nTRV+xSnfFKHqlzrHAwxf4V4W/BLaMrLPjkfoRgxWr9m+FM1XawohF
# LHQHc0AA8HFHZMhKr48WtcoSDgvc/T3PwaEBAHtoZn0DXVnec5V+vVFs45YcyojC
# r2F5/zWVka2MLzmvd/i6liVQz43mjWqQJKze/nOv+vT1XpEOC+DRxUEyt/efNIiI
# 05prArcx3Sdq/JOAdqtX2q8gp7AMKuUa2mYIL5GLgcnc/8hbevZMmsZ2myNk4iJN
# TzIO/7s1bKZ6W7HSqktXl+uepbpVgw==
# SIG # End signature block
