##############################################################################
##
## Load-Zen-Datatools.ps1 [install|remove}
##
## by Actian Corporation (http://www.actian.com/zen)
##
##############################################################################

    <#
    .NOTES
        Copyright (c) Actian Corporation.  All rights reserved.

        Use of this script is subject to the terms of the Actian Zen license
        agreement under which you licensed the Zen product.  If you did not
        accept the terms of the license agreement, you are not authorized to
        use this script. For the terms of the license, please see the license
        agreement between you and Actian Corporation.
        THE SCRIPT IS PROVIDED "AS IS", WITH NO WARRANTIES.

    .SYNOPSIS
        Integrates the Data Tools for the PSQL ADO.NET Provider v4.6 into
        Visual Studio 2019 or later. Also removes the integration from Visual
        Studio 2019 or later.

    .DESCRIPTION
        Locates the latest installed version of Visual Studio (after VS 2019)
        using the vswhere.exe utility then either installs or removes the
        appropriate Visual Studio Extension package for the Data Tools for PSQL
        ADO.NET Provider v4.6. If Visual Studio is configured for Entity Framework
        development, also copies or deletes the Entity Framework Model First
        templates for the PSQL ADO.NET Provider v4.6 to/from the appropriate
        Visual Studio extensions folder.

    .PARAMETER install
        Uses the VSIXInstaller.exe utility to install the appropriate PSQL ADO.NET
        Provider Data Tools VSIX package. Also copies the Entity Framework Model
        First templates to the appropriate Visual Studio 2019 or later installation
        location if Visual Studio is configured for Entity Framework development.
        A log file for the VSIXInstaller utility output is created in the %TEMP%
        folder with the name "zen.datatools.vsix.install.log".

    .PARAMETER remove
        Uses the VSIXInstaller.exe utility to uninstall the PSQL ADO.NET Provider
        Data Tools VSIX package. Also deletes the Entity Framework Model First
        templates from the Visual Studio 2019 or later installation location if
        Visual Studio is configured for Entity Framework development. A log file
        for the VSIXInstaller utility output is created in the %TEMP% folder with
        the name "zen.datatools.vsix.uninstall.log".

    .EXAMPLE
        PS C:\> .Load-Zen-Datatools.ps1 install
            Installs the VSIX package and copies the Entity Framework Model First
            templates to Visual Studio 2019 or later.

    .EXAMPLE
        PS C:\> .\Load-Zen-Datatools.ps1 remove
            Removes the installed VSIX package and deletes the Entity Framework Model
            First templates from Visual Studio 2019 or later.

    .INPUTS
        None. You cannot pipe objects to this cmdlet.

    .OUTPUTS
        0 : The script completed with no errors (may contain warnings).
        -1: The script returned one or more errors.

    #>

#Requires -version 3.0

param (
    [Parameter(
        Mandatory=$true,
        Position = 0)]
    [ValidateSet('install','remove')]
    [String]
    $InstallMode
)

Set-StrictMode -Version Latest

Import-LocalizedData msg -FileName "Load-Zen-DataTools.psd1" -ErrorAction Ignore

## Check for administrator rights without using the -RunAsAdministrator parameter for the #Requires
## directive (which requires PowerShell 4 or later).
$CurrentUser = New-Object -TypeName Security.Principal.WindowsPrincipal -ArgumentList $([Security.Principal.WindowsIdentity]::GetCurrent())
if ( ! $CurrentUser.IsInRole([Security.Principal.WindowsBuiltinRole]::Administrator)) {Throw $msg.errMustBeAdmin}

## Check PowerShell process is 64-bit on 64-bit OS.
$is32on64=$env:PROCESSOR_ARCHITEW6432 # Only defined if running 32-bit PS on 64-bit Windows.
if ($is32on64) { Throw $msg.errMustBePS64onWin64 }

## Global variable - exists as long as PoSh session is running.
$global:ZenInfo=""

## Script-level variables definitions.
$scriptName = & {$myInvocation.ScriptName}
$scriptPath = & {Split-Path $myInvocation.ScriptName}
$registryPath = "HKLM:\SOFTWARE\Actian\Zen"
$supportedZenMajorVersionLevel = "16"
# Note the vsixPkgName* and vsixId* vars must match the associated VS version checks in GetVSInfo{}.
$vsixPkgName1 = Join-Path $scriptPath 'Zen.VisualStudio.DataTools_2019.vsix'
$vsixId1='PSQLDataTools.d115023a-8951-4aa5-ae36-94d7b1e6f6ad'  # Zen.VisualStudio.DataTools_2019.vsix
$vsixPkgName2 = Join-Path $scriptPath 'Zen.VisualStudio.DataTools_2022.vsix'
$vsixId2='PSQLDataTools.d115023a-8951-4aa5-ae36-94d7b1e6f6ae' # Zen.VisualStudio.DataTools_2022.vsix
# --
$modelFirstIncNames='GenerateZen.Utility*.ttinclude' #ModelFirst Include Template Names
$modelFirstT4Names='SSDLtoZen*.tt' # ModelFirst T4 Template Names
$modelFirstIncludes = join-path $scriptPath "Templates\ModelFirstTemplates\$modelFirstIncNames"
$modelFirstT4 = join-path $scriptPath "Templates\ModelFirstTemplates\$modelFirstT4Names"
$vswhere=join-path $scriptPath 'util\vswhere.exe'
$relPathVSIXInstaller='Common7\IDE\VSIXInstaller.exe'
$relPathEntityFrameworkTools='Common7\IDE\Extensions\Microsoft\Entity Framework Tools'
$relPathEntityFrameworkIncludes=join-path $relPathEntityFrameworkTools 'Templates\Includes'
$relPathEntityFrameworkT4=join-path $relPathEntityFrameworkTools 'DBGen'

function Main {
    ##################################################################################################
    ## Invoked as entry-point function at end of script and defines the order the other defined
    ## functions are called.
    ##################################################################################################
    if ($InstallMode -eq "install") {
        FindProductType
        CheckZenVersion
    }
    GetVSInfo
    RunVsixInstaller
    CopyOrRemoveTemplates
}

function FindProductType {
    # NOTE: This function is duplicated exactly as-is in the Zen Edge WinUWP uninstall script Uninstall-Zen.ps1
    # Any changes here should be kept in-sync with the other script (starting below this line).
    ##########################
    # Sets the installed product type based on one the 'HKLM:\SOFTWARE\Actian\Zen\InstallInfo'
    # sub-keys for Client, Server, Workgroup.
    #
    # Variables set:
    #   [string] $script:InstalledProductType = client|server|reporting
    #   [object] $global:ZenInfo = object methods match InstallInfo registry values.
    #                   $ZenInfo.InstallData
    #                   $ZenInfo.InstallDir64
    #                   $ZenInfo.InstallDir
    #                   $ZenInfo.IsBtrieve
    #                   $ZenInfo.IsReport
    #                   $ZenInfo.IsVx
    #                   $ZenInfo.ProductVersion
    #                   $ZenInfo.VersionLevel
    #                   $ZenInfo.InstallData
    #
    # Note: Must be called before logging is started.
    ##########################
    $script:InstalledProductType=""
    if (test-path $registryPath"\InstallInfo\") {
        $prodName = Get-ChildItem -Name $registryPath"\InstallInfo\"
        Get-ChildItem $registryPath"\InstallInfo\" | ForEach-Object -Process {
        Write-Host "Name: "  $prodName
            switch ($prodName) {
                'Client' {
                    $global:ZenInfo=Get-ItemProperty $registryPath"\InstallInfo\$prodName"
                    $script:InstalledProductType="client"
                    break}
                'Server' {
                    $global:ZenInfo=Get-ItemProperty $registryPath"\InstallInfo\$prodName"
                    if ($ZenInfo.IsVx -eq 0) {
                        $script:InstalledProductType="server"
                    } else {
                        $script:InstalledProductType="vx"
                    }
                    break}
                'Workgroup' {
                    $global:ZenInfo=Get-ItemProperty $registryPath"\InstallInfo\$_"
                    # IsReport is undefined for WG and PoSh throws error if object member not found.
                    $script:InstalledProductType="workgroup" #default
                    Try {
                        if ($global:ZenInfo.IsReport -eq 1) {
                            $script:InstalledProductType="reporting"
                        } 
                    } Catch {}
                    break}
            }
        }
    }
}

function CheckZenVersion {
    ##################################################################################################
    ## Verifies the major version specified in $ZenInfo.ProductVersion (global object set by previous
    ## call to FindProductType) starts with the major version number defined for by script variable
    ## $supportedZenMajorVersionLevel. The script is aborted if the detected major version is not
    ## the same as the defined major version.
    ##################################################################################################
    if ($ZenInfo -ne "") {
        $msg.infoZenInfo -f $ZenInfo.ProductVersion| Write-Output
        $currentVersion=$ZenInfo.ProductVersion[0]+$ZenInfo.ProductVersion[1]
        if ("$supportedZenMajorVersionLevel" -gt $currentVersion ) {
           $msg.errWrongZenVer -f $supportedZenMajorVersionLevel | Write-Error;exit -1
        }
    } else {
        $msg.errNoZenFound | Write-Error;exit -1
    }
}

function GetVSInfo {
    ##################################################################################################
    ## Creates the script level PoSh object "vsInfo" with the member properties specified below. If
    ## Visual Studio 2019 or later is not detected the script displays an error message and aborts.
    ## If the retrieved path to the VSIXInstaller utility is invalid the script also displays an error
    ## message and aborts.
    ##
    ## object member properties:
    ##    vsInfo.installDir    : value of property 'installationPath' returned by vswhere utility.
    ##    vsInfo.model1_inc    : path to Entity Framework Tools\Templates\Includes folder if it exists,
    ##                           empty string if it does not exist.
    ##    vsInfo.model1_t4     : path to Entity Framework Tools\DBGen folder if it exists, empty string
    ##                           if it does not exist.
    ##    vsInfo.version       : value of property 'installationVersion' returned by vswhere utility.
    ##    vsInfo.vsixInstaller : path to the VSIXInstaller utility (must exist).
    ##################################################################################################
    $vsInstallDirFound=$false

    ## Get installed version and location of VS. See the following URL for information on valid version
    ## specifiers for VS: https://docs.microsoft.com/en-us/visualstudio/extensibility/vsix-extension-schema-2-0-reference

    ## Examples to obtain specific version ranges.
    # Visual Studio 2022: version "[17.0,18.0)"
    # $vsVersion=& $vswhere -nologo -products * -property installationVersion -version [17.0,18.0)

    # Visual Studio 2019: version "[16.0,17.0)"
    # $vsVersion=& $vswhere -nologo -products * -property installationVersion -version [16.0,17.0)

    # Visual Studio 2017: version "[15.0,16.0)"
    # $vsVersion=& $vswhere -nologo -products * -property installationVersion -version [15.0,16.0)

    # Just get the latest installed version.  This matches VS 2017 and later because VS 2015 and
    # earlier require the "-legacy" option to retrieve the installationVersion property so without
    # that option specified VS 2017 is the earliest that would be detected even if VS 2015 or earlier
    # is also installed.
    $vsVersion=& $vswhere -nologo -latest -products * -property installationVersion
    if ($vsVersion) {
        ## VS 2017 or later.
        $vsInstallDir=& $vswhere -nologo -latest -products * -property installationPath
        if ($vsInstallDir -and (Test-Path -Path $vsInstallDir)) {$vsInstallDirFound=$True}
    }

    ## Abort script if $vsVersion is not "16.0" or later (Visual Studio 2019 or later) or if the
    ## installation path does not exist.
    if ( (!$vsVersion) -or ($vsVersion -lt "16.0") -or (!$vsInstallDirFound) ) {$msg.errNoVSFound -f "2019"| Write-Error;exit -1}

    ## Verify the location of the VSIXInstaller utility required to install/remove the VSIX package.
    $vsixInstaller=Join-Path $vsInstallDir $relPathVSIXInstaller
    if (! (test-path $vsixInstaller)) {$msg.errNoVSIXInstaller | Write-Error;exit -1}

    ## Verify the location of the Entity Framework Tools "Templates\Includes" and "DBGen" folders
    ## (sets corresponding vsInfo properties to empty string if the folder doesn't exist).
    $entityFrameworkIncludesPath=Join-Path $vsInstallDir $relPathEntityFrameworkIncludes
    if (!(test-path $entityFrameworkIncludesPath)) {$entityFrameworkIncludesPath=""}
    $entityFrameworkDBGenPath=Join-Path $vsInstallDir $relPathEntityFrameworkT4
    if (!(test-path $entityFrameworkDBGenPath)) {$entityFrameworkDBGenPath=""}

    ## If the script has gotten this far VS 2019 or later is installed.  We have different VSIX
    ## packages for different versions of VS so we need to know which one to use.
    if ( ($vsVersion -lt "17.0" ) ) {
        ## VS 2019
        $vsixPkgToUse=$vsixPkgName1
        $vsixIdToUse=$vsixId1
    } else {
        ## VS 2022 or later
        $vsixPkgToUse=$vsixPkgName2
        $vsixIdToUse=$vsixId2
    }

    ## Create PoSh object vsInfo.
    $script:vsInfo = @{
        'installDir'=$vsInstallDir;
        'model1_inc'=$entityFrameworkIncludesPath;
        'model1_t4'=$entityFrameworkDBGenPath;
        'version'=$vsVersion;
        'vsixInstaller'=$vsixInstaller;
        'vsixPackage'=$vsixPkgToUse;
        'vsixID'=$vsixIdToUse;
    }

    ## Output object contents.
    $script:vsInfo | Write-Output

    ## Output VS version and location info.
    $msg.infoVisualStudioInfo -f $vsInfo.version,$vsInfo.installDir | Write-Output
}

function RunVsixInstaller {
    ##################################################################################################
    ## Runs VSIXInstaller utility to install or remove the appropriate VSIX package.  If the
    ## VSIXInstaller.exe process cannot be started an error message is displayed and the script is
    ## aborted, otherwise an informational message instructing the user to check the VSIXInstaller log
    ## file is displayed.  The VSIX package name and package ID are stored in the vsInfo object
    ## created by GetVSInfo{}.
    ##################################################################################################

    ## Set the VSIX package name and install/uninstall options based on Visual Studio version.
    $vsixInstallArgs="/quiet","/logFile:Zen.DataTools.VSIX.install.log",$vsInfo.vsixPackage
    $vsixInstallMsg=$msg.infoInstalledVsixPkgLatestWithLog -f [string]"Zen.DataTools.VSIX.install.log"
    $vsixInstallErr=$msg.errVsixInstallWithLog -f [string]" (Zen.DataTools.VSIX.install.log)"
    $vsixRemovalArgs="/quiet","/logFile:Zen.DataTools.VSIX.uninstall.log",("/uninstall:"+$vsInfo.vsixID)
    $vsixRemovalMsg=$msg.infoRemovedVsixPkgLatestWithLog -f [string]"Zen.DataTools.VSIX.uninstall.log"
    $vsixRemovalErr=$msg.errVsixUninstallWithLog -f [string]" (Zen.DataTools.VSIX.uninstall.log)"


    ## Install or remove the VSIX package based on $InstallMode.
    if ($InstallMode -eq "install") {
        Try {
           $msg.infoInstallingVsixPkg | Write-Output
           Start-Process -FilePath $vsInfo.vsixInstaller -ArgumentList $vsixInstallArgs -NoNewWindow -Wait
           $vsixInstallMsg | Write-Output
        }
        Catch {$vsixInstallErr | Write-Error;exit -1}
    } else {
        Try {
            $msg.infoRemovingVsixPkg | Write-Output
            Start-Process -FilePath $vsInfo.vsixInstaller -ArgumentList $vsixRemovalArgs -NoNewWindow -Wait
            $vsixRemovalMsg | Write-Output
        }
        Catch {$vsixRemovalErr | Write-Error;exit -1}
    }
}

function CopyOrRemoveTemplates {
    ##################################################################################################
    ## Copies or removes the Entity Framework Model First templates to/from the appropriate Visual
    ## Studio folder.  If the path to the Entity Framework Tools folder cannot be located this step
    ## is skipped.
    ##################################################################################################

    if ($InstallMode -eq "install") {  ## Copy templates.
        Try {
            ## Verify Entity Framework Tools folder and copy templates.
            $entityFrameworkTools=join-path $vsInfo.installDir $relPathEntityFrameworkTools
            if (Test-Path $entityFrameworkTools) {

                ## Model First include templates.
                Copy-Item -Path $modelFirstIncludes -Destination $vsInfo.model1_inc -Force
                if ($?) {$msg.infoAddedModelFirstIncludeTemplates | Write-Output}

                ## Model First T4 templates.
                Copy-Item -Path $modelFirstT4 -Destination $vsInfo.model1_t4 -Force
                if ($?) {$msg.infoAddedModelFirstT4Templates | Write-Output}

            }
        }
        Catch {$msg.warnNoTemplates -f [string]"$relPathEntityFrameworkTools" | Write-Warning}
    } else {  ## Remove templates.
        Try {
            ## Verify Model First include templates exist and remove them.
            $removeIncludes=join-path $vsInfo.model1_inc $modelFirstIncNames
            if (Test-Path $removeIncludes) {
                remove-item -Path $removeIncludes -Force
                if ($?) {$msg.infoRemovedModelFirstIncludeTemplates | Write-Output}
            }

            ## Verify Model First T4 templates exist and remove them.
            $removeT4=join-path $vsInfo.model1_t4 $modelFirstT4Names
            if (Test-Path $removeT4) {
                remove-item -Path $removeT4 -Force
                if ($?) {$msg.infoRemovedModelFirstT4Templates | Write-Output}
            }
        }
        Catch {$msg.warnNoTemplates -f [string]"$relPathEntityFrameworkTools" | Write-Warning}
    }
}

## Invoke Main function.
"" | Write-Output
$msg.infoRunning -f $MyInvocation.InvocationName, $InstallMode | Write-Output
. Main
$msg.infoEnd -f $MyInvocation.InvocationName, $InstallMode | Write-Output
"" | Write-Output
exit 0
# SIG # Begin signature block
# MIIpUwYJKoZIhvcNAQcCoIIpRDCCKUACAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCBTMYx88qIxFLka
# 9hvnlsXxcZP76j5sL/dEofwgy/znb6CCDiowggawMIIEmKADAgECAhAIrUCyYNKc
# TJ9ezam9k67ZMA0GCSqGSIb3DQEBDAUAMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQK
# EwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNV
# BAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0yMTA0MjkwMDAwMDBaFw0z
# NjA0MjgyMzU5NTlaMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAw
# ggIKAoICAQDVtC9C0CiteLdd1TlZG7GIQvUzjOs9gZdwxbvEhSYwn6SOaNhc9es0
# JAfhS0/TeEP0F9ce2vnS1WcaUk8OoVf8iJnBkcyBAz5NcCRks43iCH00fUyAVxJr
# Q5qZ8sU7H/Lvy0daE6ZMswEgJfMQ04uy+wjwiuCdCcBlp/qYgEk1hz1RGeiQIXhF
# LqGfLOEYwhrMxe6TSXBCMo/7xuoc82VokaJNTIIRSFJo3hC9FFdd6BgTZcV/sk+F
# LEikVoQ11vkunKoAFdE3/hoGlMJ8yOobMubKwvSnowMOdKWvObarYBLj6Na59zHh
# 3K3kGKDYwSNHR7OhD26jq22YBoMbt2pnLdK9RBqSEIGPsDsJ18ebMlrC/2pgVItJ
# wZPt4bRc4G/rJvmM1bL5OBDm6s6R9b7T+2+TYTRcvJNFKIM2KmYoX7BzzosmJQay
# g9Rc9hUZTO1i4F4z8ujo7AqnsAMrkbI2eb73rQgedaZlzLvjSFDzd5Ea/ttQokbI
# YViY9XwCFjyDKK05huzUtw1T0PhH5nUwjewwk3YUpltLXXRhTT8SkXbev1jLchAp
# QfDVxW0mdmgRQRNYmtwmKwH0iU1Z23jPgUo+QEdfyYFQc4UQIyFZYIpkVMHMIRro
# OBl8ZhzNeDhFMJlP/2NPTLuqDQhTQXxYPUez+rbsjDIJAsxsPAxWEQIDAQABo4IB
# WTCCAVUwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQUaDfg67Y7+F8Rhvv+
# YXsIiGX0TkIwHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4cD08wDgYDVR0P
# AQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHcGCCsGAQUFBwEBBGswaTAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsGAQUFBzAC
# hjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9v
# dEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2VydC5j
# b20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAcBgNVHSAEFTATMAcGBWeBDAED
# MAgGBmeBDAEEATANBgkqhkiG9w0BAQwFAAOCAgEAOiNEPY0Idu6PvDqZ01bgAhql
# +Eg08yy25nRm95RysQDKr2wwJxMSnpBEn0v9nqN8JtU3vDpdSG2V1T9J9Ce7FoFF
# UP2cvbaF4HZ+N3HLIvdaqpDP9ZNq4+sg0dVQeYiaiorBtr2hSBh+3NiAGhEZGM1h
# mYFW9snjdufE5BtfQ/g+lP92OT2e1JnPSt0o618moZVYSNUa/tcnP/2Q0XaG3Ryw
# YFzzDaju4ImhvTnhOE7abrs2nfvlIVNaw8rpavGiPttDuDPITzgUkpn13c5Ubdld
# AhQfQDN8A+KVssIhdXNSy0bYxDQcoqVLjc1vdjcshT8azibpGL6QB7BDf5WIIIJw
# 8MzK7/0pNVwfiThV9zeKiwmhywvpMRr/LhlcOXHhvpynCgbWJme3kuZOX956rEnP
# LqR0kq3bPKSchh/jwVYbKyP/j7XqiHtwa+aguv06P0WmxOgWkVKLQcBIhEuWTatE
# QOON8BUozu3xGFYHKi8QxAwIZDwzj64ojDzLj4gLDb879M4ee47vtevLt/B3E+bn
# KD+sEq6lLyJsQfmCXBVmzGwOysWGw/YmMwwHS6DTBwJqakAwSEs0qFEgu60bhQji
# WQ1tygVQK+pKHJ6l/aCnHwZ05/LWUpD9r4VIIflXO7ScA+2GRfS0YW6/aOImYIbq
# yK+p/pQd52MbOoZWeE4wggdyMIIFWqADAgECAhALug4gRzX+ySpZsBzdR9OMMA0G
# CSqGSIb3DQEBCwUAMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwHhcNMjMwNDI0MDAwMDAwWhcNMjYwNDIz
# MjM1OTU5WjB6MQswCQYDVQQGEwJVUzEOMAwGA1UECBMFVGV4YXMxEzARBgNVBAcT
# ClJvdW5kIFJvY2sxGzAZBgNVBAoTEkFjdGlhbiBDb3Jwb3JhdGlvbjEMMAoGA1UE
# CxMDWmVuMRswGQYDVQQDExJBY3RpYW4gQ29ycG9yYXRpb24wggIiMA0GCSqGSIb3
# DQEBAQUAA4ICDwAwggIKAoICAQCmrbJewixh8zYkjt5G34WYugoGnAfUHCzJmXGn
# sNqZ7jSNZR8Fy3Z9aFFUkwYULY2CQxNNC/GSdnt1lB0TM1ggSido3ttaJ2FaUD9F
# oi1keyc9pR8cKjtt8R1go+BNHy4MIPrr6HbI9ObXxGwP5WVAW72jaOW2rJcGOKuB
# wG642zwhAA4wJxRJKV394PxuUsLmqqyarb8wdvAhZJUP0OZiV5ERJ0ZglBv4qdPt
# e0u4pZCpbhhlPEptFHX2bEdY9eKAeqZClenQyzJqJkOgpeGPjoN8mh1FLeLlwSjs
# AdiAL4KBHyAQb2kGEU7uJaKpyoq1VxSerNWtZUmihQMP9EJg7cHSZPJNj2HBhhOR
# r1NbEzyZjx2I1G/Ib69A0R6CaiEXyJnLJxg4R5VazeM87YS23ML0jFyDiCzfnQp5
# W+ls8i0MHhqqVqpsRZPViHHjJtNpcAR7KsbRrlsXL9c5kMqNvKxDvZeV0oVSECAd
# 3Djtr0pfSvXU0eKIl8i3LDimc+VcTtbbfs93TGDyVaNb9uDTlAKhk46PaYhamNqr
# ZCkWWiNSdnglito6wKaOa91AnBN2WbQxOEEpz8UEvNPYM/+HGYIwefXspJz4wvAB
# yB9AYShQ4oUGPnQ/+fPWZFZ/yTi7Gpb69FedVltEiL7A+3Ds7UKZ/269YhTQYSGh
# P5R8HQIDAQABo4ICAzCCAf8wHwYDVR0jBBgwFoAUaDfg67Y7+F8Rhvv+YXsIiGX0
# TkIwHQYDVR0OBBYEFLZN9fT8GcYstZ/eHlOoSwqtoRPtMA4GA1UdDwEB/wQEAwIH
# gDATBgNVHSUEDDAKBggrBgEFBQcDAzCBtQYDVR0fBIGtMIGqMFOgUaBPhk1odHRw
# Oi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRDb2RlU2lnbmlu
# Z1JTQTQwOTZTSEEzODQyMDIxQ0ExLmNybDBToFGgT4ZNaHR0cDovL2NybDQuZGln
# aWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0Q29kZVNpZ25pbmdSU0E0MDk2U0hB
# Mzg0MjAyMUNBMS5jcmwwPgYDVR0gBDcwNTAzBgZngQwBBAEwKTAnBggrBgEFBQcC
# ARYbaHR0cDovL3d3dy5kaWdpY2VydC5jb20vQ1BTMIGUBggrBgEFBQcBAQSBhzCB
# hDAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMFwGCCsGAQUF
# BzAChlBodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVk
# RzRDb2RlU2lnbmluZ1JTQTQwOTZTSEEzODQyMDIxQ0ExLmNydDAJBgNVHRMEAjAA
# MA0GCSqGSIb3DQEBCwUAA4ICAQCWGybROBuJGQ2OT6P6pJtQptFx2fprQP+Ttb5L
# 97xCPKnI72HljOJUKIiC+QJCqolKBFvtxbtzsFap4fXkBQVsDcrNijMLjX5IxAFv
# l2N7kQ03POGihblkie2n+HgX0Frnd9PrJxSBalzylmsNtHCvhgwdnSZzPBb9pGW8
# zLJjsEiFGOPvH6UP2TfLNZHpys10DpM5U44mw9h034EHuNafF4Dbxp7heu58jtWS
# QDzFmVe2+4/tk5eHAHx0IuhAWYhtTlygMZK54oOp0DHSRnQKMfeqa7XUS5pwVFR1
# MLbZETlr8SS2Y5AnPHKGE4vFOVXOznwEmYUTj/+bRnL0cm7ndeLsAfHVQsJjmF4K
# gcNaqMaAuAV80M4LrFt+bV0JUC83cu2RmrMKQyO1FEm7yp7VcQniVIbwBv1w+rOh
# aKAW+6Zoj4n6QrpYpdIdjs5gY2crwzUs5wst16bkVHQh+vFX6KCax41RcXTvLTbI
# T4Hnz3zSOTQKLHunfQs4/QyUd8LoAPnKJGH9Dwgs3BESIbFcdLDuzQ9Gza9dUaBX
# eswMXJug5CaO8gblx99JD6jZBGV6hUb9+72UTKWy+3Yv2KkFPmPLxUqyrAoG+MR9
# DOEeMZP10Qjfu7OeRZoR6oSeWLQJUUS2zybxFq4Xxx6C5v58ZUtO9gOjLhrwB72m
# mPuBWDGCGn8wghp7AgEBMH0waTELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lD
# ZXJ0LCBJbmMuMUEwPwYDVQQDEzhEaWdpQ2VydCBUcnVzdGVkIEc0IENvZGUgU2ln
# bmluZyBSU0E0MDk2IFNIQTM4NCAyMDIxIENBMQIQC7oOIEc1/skqWbAc3UfTjDAN
# BglghkgBZQMEAgEFAKCBljAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAqBgorBgEEAYI3AgEMMRwwGqEYgBZo
# dHRwOlxcd3d3LmFjdGlhbi5jb20gMC8GCSqGSIb3DQEJBDEiBCCsuzMwhxKncyhK
# 9QwovsQuCT56uNIKS2ZkQ9xWMCr9eDANBgkqhkiG9w0BAQEFAASCAgB5xrkaAnLj
# qloZcK7AGy8Smh5/o4g++OS2y07UwTQ4i6tnygjAtAlZopLNxRDhBT5FFfK1epS4
# I8QpDWDdMeSr11DF6v8LD9xPhmxaF8AXsokzENHGWOBy+NKWgsl5WxRYYWDxUSJN
# Fn6+vGRoixkF5Si/VuYEWNkLw9hhk2MV22hUCl1qxS1lnPhl7v3XARuNGHEWGvl0
# AA+J3tbdBy0gCf5PehWUMrEUqtdOS0iOv790YKmPOsHAzNuMl7TNEr97lzbT44Kd
# rS1ntpCVSjOjz26xq6ZQUS0TfTaK7U4hBl8oU3+xl1EMntzwwA40NORFG2gswwVi
# HiP/mORFiUfuUBWvP1+wBJF6S4M8l4e2Qk9nWvUokt869d2hyuqGlddhgafLFebZ
# KzP2coHERJAqQXctb6QHJiOzcMUWeCLJHtN1luqLQPDuHJLUX/cTi6TeCabjAvCY
# nt+ZfzkxENXAiZ+0JWucmNAJvLeHVeRO//bEichtN0IYdTB3Sg220JcDsnY1RHWE
# zniZX1tUn8PSiBdiaKX7U/Y9GLWS0mjiN/WCUd7u+QEITnjjPzs9d8bxct46NY3f
# w3ODTPHAIBWrXTQslMC4WupBY+Vd1tLtx2hdvgl82W3xr1TZXKke8EIU2fnL9ZLG
# a4zXeHSVxsxeTsLMD8Byv59UT3RDQ3mu0KGCFzowghc2BgorBgEEAYI3AwMBMYIX
# JjCCFyIGCSqGSIb3DQEHAqCCFxMwghcPAgEDMQ8wDQYJYIZIAWUDBAIBBQAweAYL
# KoZIhvcNAQkQAQSgaQRnMGUCAQEGCWCGSAGG/WwHATAxMA0GCWCGSAFlAwQCAQUA
# BCBpwWH1n6XR0XLlvXtWDX+DnDgj9BUvMHoqYiorrtw9kwIRANSSeUhSiwKk2xBX
# 40r7Q3gYDzIwMjUwNjI3MDEwOTE2WqCCEwMwgga8MIIEpKADAgECAhALrma8Wrp/
# lYfG+ekE4zMEMA0GCSqGSIb3DQEBCwUAMGMxCzAJBgNVBAYTAlVTMRcwFQYDVQQK
# Ew5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3RlZCBHNCBS
# U0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwHhcNMjQwOTI2MDAwMDAwWhcN
# MzUxMTI1MjM1OTU5WjBCMQswCQYDVQQGEwJVUzERMA8GA1UEChMIRGlnaUNlcnQx
# IDAeBgNVBAMTF0RpZ2lDZXJ0IFRpbWVzdGFtcCAyMDI0MIICIjANBgkqhkiG9w0B
# AQEFAAOCAg8AMIICCgKCAgEAvmpzn/aVIauWMLpbbeZZo7Xo/ZEfGMSIO2qZ46XB
# /QowIEMSvgjEdEZ3v4vrrTHleW1JWGErrjOL0J4L0HqVR1czSzvUQ5xF7z4IQmn7
# dHY7yijvoQ7ujm0u6yXF2v1CrzZopykD07/9fpAT4BxpT9vJoJqAsP8YuhRvflJ9
# YeHjes4fduksTHulntq9WelRWY++TFPxzZrbILRYynyEy7rS1lHQKFpXvo2GePfs
# MRhNf1F41nyEg5h7iOXv+vjX0K8RhUisfqw3TTLHj1uhS66YX2LZPxS4oaf33rp9
# HlfqSBePejlYeEdU740GKQM7SaVSH3TbBL8R6HwX9QVpGnXPlKdE4fBIn5BBFnV+
# KwPxRNUNK6lYk2y1WSKour4hJN0SMkoaNV8hyyADiX1xuTxKaXN12HgR+8WulU2d
# 6zhzXomJ2PleI9V2yfmfXSPGYanGgxzqI+ShoOGLomMd3mJt92nm7Mheng/TBeSA
# 2z4I78JpwGpTRHiT7yHqBiV2ngUIyCtd0pZ8zg3S7bk4QC4RrcnKJ3FbjyPAGogm
# oiZ33c1HG93Vp6lJ415ERcC7bFQMRbxqrMVANiav1k425zYyFMyLNyE1QulQSgDp
# W9rtvVcIH7WvG9sqYup9j8z9J1XqbBZPJ5XLln8mS8wWmdDLnBHXgYly/p1DhoQo
# 5fkCAwEAAaOCAYswggGHMA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBYG
# A1UdJQEB/wQMMAoGCCsGAQUFBwMIMCAGA1UdIAQZMBcwCAYGZ4EMAQQCMAsGCWCG
# SAGG/WwHATAfBgNVHSMEGDAWgBS6FtltTYUvcyl2mi91jGogj57IbzAdBgNVHQ4E
# FgQUn1csA3cOKBWQZqVjXu5Pkh92oFswWgYDVR0fBFMwUTBPoE2gS4ZJaHR0cDov
# L2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0UlNBNDA5NlNIQTI1
# NlRpbWVTdGFtcGluZ0NBLmNybDCBkAYIKwYBBQUHAQEEgYMwgYAwJAYIKwYBBQUH
# MAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBYBggrBgEFBQcwAoZMaHR0cDov
# L2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0UlNBNDA5NlNI
# QTI1NlRpbWVTdGFtcGluZ0NBLmNydDANBgkqhkiG9w0BAQsFAAOCAgEAPa0eH3aZ
# W+M4hBJH2UOR9hHbm04IHdEoT8/T3HuBSyZeq3jSi5GXeWP7xCKhVireKCnCs+8G
# Zl2uVYFvQe+pPTScVJeCZSsMo1JCoZN2mMew/L4tpqVNbSpWO9QGFwfMEy60HofN
# 6V51sMLMXNTLfhVqs+e8haupWiArSozyAmGH/6oMQAh078qRh6wvJNU6gnh5OruC
# P1QUAvVSu4kqVOcJVozZR5RRb/zPd++PGE3qF1P3xWvYViUJLsxtvge/mzA75oBf
# FZSbdakHJe2BVDGIGVNVjOp8sNt70+kEoMF+T6tptMUNlehSR7vM+C13v9+9ZOUK
# zfRUAYSyyEmYtsnpltD/GWX8eM70ls1V6QG/ZOB6b6Yum1HvIiulqJ1Elesj5TMH
# q8CWT/xrW7twipXTJ5/i5pkU5E16RSBAdOp12aw8IQhhA/vEbFkEiF2abhuFixUD
# obZaA0VhqAsMHOmaT3XThZDNi5U2zHKhUs5uHHdG6BoQau75KiNbh0c+hatSF+02
# kULkftARjsyEpHKsF7u5zKRbt5oK5YGwFvgc4pEVUNytmB3BpIiowOIIuDgP5M9W
# ArHYSAR16gc0dP2XdkMEP5eBsX7bf/MGN4K3HP50v/01ZHo/Z5lGLvNwQ7XHBx1y
# omzLP8lx4Q1zZKDyHcp4VQJLu2kWTsKsOqQwggauMIIElqADAgECAhAHNje3JFR8
# 2Ees/ShmKl5bMA0GCSqGSIb3DQEBCwUAMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQK
# EwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNV
# BAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0yMjAzMjMwMDAwMDBaFw0z
# NzAzMjIyMzU5NTlaMGMxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3RlZCBHNCBSU0E0MDk2IFNIQTI1
# NiBUaW1lU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoIC
# AQDGhjUGSbPBPXJJUVXHJQPE8pE3qZdRodbSg9GeTKJtoLDMg/la9hGhRBVCX6SI
# 82j6ffOciQt/nR+eDzMfUBMLJnOWbfhXqAJ9/UO0hNoR8XOxs+4rgISKIhjf69o9
# xBd/qxkrPkLcZ47qUT3w1lbU5ygt69OxtXXnHwZljZQp09nsad/ZkIdGAHvbREGJ
# 3HxqV3rwN3mfXazL6IRktFLydkf3YYMZ3V+0VAshaG43IbtArF+y3kp9zvU5Emfv
# DqVjbOSmxR3NNg1c1eYbqMFkdECnwHLFuk4fsbVYTXn+149zk6wsOeKlSNbwsDET
# qVcplicu9Yemj052FVUmcJgmf6AaRyBD40NjgHt1biclkJg6OBGz9vae5jtb7IHe
# IhTZgirHkr+g3uM+onP65x9abJTyUpURK1h0QCirc0PO30qhHGs4xSnzyqqWc0Jo
# n7ZGs506o9UD4L/wojzKQtwYSH8UNM/STKvvmz3+DrhkKvp1KCRB7UK/BZxmSVJQ
# 9FHzNklNiyDSLFc1eSuo80VgvCONWPfcYd6T/jnA+bIwpUzX6ZhKWD7TA4j+s4/T
# Xkt2ElGTyYwMO1uKIqjBJgj5FBASA31fI7tk42PgpuE+9sJ0sj8eCXbsq11GdeJg
# o1gJASgADoRU7s7pXcheMBK9Rp6103a50g5rmQzSM7TNsQIDAQABo4IBXTCCAVkw
# EgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQUuhbZbU2FL3MpdpovdYxqII+e
# yG8wHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4cD08wDgYDVR0PAQH/BAQD
# AgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMIMHcGCCsGAQUFBwEBBGswaTAkBggrBgEF
# BQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsGAQUFBzAChjVodHRw
# Oi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNy
# dDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGln
# aUNlcnRUcnVzdGVkUm9vdEc0LmNybDAgBgNVHSAEGTAXMAgGBmeBDAEEAjALBglg
# hkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggIBAH1ZjsCTtm+YqUQiAX5m1tghQuGw
# GC4QTRPPMFPOvxj7x1Bd4ksp+3CKDaopafxpwc8dB+k+YMjYC+VcW9dth/qEICU0
# MWfNthKWb8RQTGIdDAiCqBa9qVbPFXONASIlzpVpP0d3+3J0FNf/q0+KLHqrhc1D
# X+1gtqpPkWaeLJ7giqzl/Yy8ZCaHbJK9nXzQcAp876i8dU+6WvepELJd6f8oVInw
# 1YpxdmXazPByoyP6wCeCRK6ZJxurJB4mwbfeKuv2nrF5mYGjVoarCkXJ38SNoOeY
# +/umnXKvxMfBwWpx2cYTgAnEtp/Nh4cku0+jSbl3ZpHxcpzpSwJSpzd+k1OsOx0I
# SQ+UzTl63f8lY5knLD0/a6fxZsNBzU+2QJshIUDQtxMkzdwdeDrknq3lNHGS1yZr
# 5Dhzq6YBT70/O3itTK37xJV77QpfMzmHQXh6OOmc4d0j/R0o08f56PGYX/sr2H7y
# Rp11LB4nLCbbbxV7HhmLNriT1ObyF5lZynDwN7+YAN8gFk8n+2BnFqFmut1VwDop
# hrCYoCvtlUG3OtUVmDG0YgkPCr2B2RP+v6TR81fZvAT6gt4y3wSJ8ADNXcL50CN/
# AAvkdgIm2fBldkKmKYcJRyvmfxqkhQ/8mJb2VVQrH4D6wPIOK+XW+6kvRBVK5xMO
# Hds3OBqhK/bt1nz8MIIFjTCCBHWgAwIBAgIQDpsYjvnQLefv21DiCEAYWjANBgkq
# hkiG9w0BAQwFADBlMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5j
# MRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMSQwIgYDVQQDExtEaWdpQ2VydCBB
# c3N1cmVkIElEIFJvb3QgQ0EwHhcNMjIwODAxMDAwMDAwWhcNMzExMTA5MjM1OTU5
# WjBiMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQL
# ExB3d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJv
# b3QgRzQwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQC/5pBzaN675F1K
# PDAiMGkz7MKnJS7JIT3yithZwuEppz1Yq3aaza57G4QNxDAf8xukOBbrVsaXbR2r
# snnyyhHS5F/WBTxSD1Ifxp4VpX6+n6lXFllVcq9ok3DCsrp1mWpzMpTREEQQLt+C
# 8weE5nQ7bXHiLQwb7iDVySAdYyktzuxeTsiT+CFhmzTrBcZe7FsavOvJz82sNEBf
# sXpm7nfISKhmV1efVFiODCu3T6cw2Vbuyntd463JT17lNecxy9qTXtyOj4DatpGY
# QJB5w3jHtrHEtWoYOAMQjdjUN6QuBX2I9YI+EJFwq1WCQTLX2wRzKm6RAXwhTNS8
# rhsDdV14Ztk6MUSaM0C/CNdaSaTC5qmgZ92kJ7yhTzm1EVgX9yRcRo9k98FpiHaY
# dj1ZXUJ2h4mXaXpI8OCiEhtmmnTK3kse5w5jrubU75KSOp493ADkRSWJtppEGSt+
# wJS00mFt6zPZxd9LBADMfRyVw4/3IbKyEbe7f/LVjHAsQWCqsWMYRJUadmJ+9oCw
# ++hkpjPRiQfhvbfmQ6QYuKZ3AeEPlAwhHbJUKSWJbOUOUlFHdL4mrLZBdd56rF+N
# P8m800ERElvlEFDrMcXKchYiCd98THU/Y+whX8QgUWtvsauGi0/C1kVfnSD8oR7F
# wI+isX4KJpn15GkvmB0t9dmpsh3lGwIDAQABo4IBOjCCATYwDwYDVR0TAQH/BAUw
# AwEB/zAdBgNVHQ4EFgQU7NfjgtJxXWRM3y5nP+e6mK4cD08wHwYDVR0jBBgwFoAU
# Reuir/SSy4IxLVGLp6chnfNtyA8wDgYDVR0PAQH/BAQDAgGGMHkGCCsGAQUFBwEB
# BG0wazAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMGCCsG
# AQUFBzAChjdodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1
# cmVkSURSb290Q0EuY3J0MEUGA1UdHwQ+MDwwOqA4oDaGNGh0dHA6Ly9jcmwzLmRp
# Z2ljZXJ0LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcmwwEQYDVR0gBAow
# CDAGBgRVHSAAMA0GCSqGSIb3DQEBDAUAA4IBAQBwoL9DXFXnOF+go3QbPbYW1/e/
# Vwe9mqyhhyzshV6pGrsi+IcaaVQi7aSId229GhT0E0p6Ly23OO/0/4C5+KH38nLe
# JLxSA8hO0Cre+i1Wz/n096wwepqLsl7Uz9FDRJtDIeuWcqFItJnLnU+nBgMTdydE
# 1Od/6Fmo8L8vC6bp8jQ87PcDx4eo0kxAGTVGamlUsLihVo7spNU96LHc/RzY9Hda
# XFSMb++hUD38dglohJ9vytsgjTVgHAIDyyCwrFigDkBjxZgiwbJZ9VVrzyerbHbO
# byMt9H5xaiNrIv8SuFQtJ37YOtnwtoeW/VvRXKwYw02fc7cBqZ9Xql4o4rmUMYID
# djCCA3ICAQEwdzBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIElu
# Yy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBTSEEyNTYg
# VGltZVN0YW1waW5nIENBAhALrma8Wrp/lYfG+ekE4zMEMA0GCWCGSAFlAwQCAQUA
# oIHRMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAcBgkqhkiG9w0BCQUxDxcN
# MjUwNjI3MDEwOTE2WjArBgsqhkiG9w0BCRACDDEcMBowGDAWBBTb04XuYtvSPnvk
# 9nFIUIck1YZbRTAvBgkqhkiG9w0BCQQxIgQgpyjk19P5miRZD+qEWjAp/P/yszcg
# up2R3fYCoBTt850wNwYLKoZIhvcNAQkQAi8xKDAmMCQwIgQgdnafqPJjLx9DCzoj
# MK7WVnX+13PbBdZluQWTmEOPmtswDQYJKoZIhvcNAQEBBQAEggIASGzYlQtzrRGw
# QEj8n5qYM7iEN/fERCK9i1DZQESYzi6uK6EPkFCQ0gYr8Q+2j2lBrV9rav9MGxNm
# MstuJh6CbwF0zblVzEL8U5OyOf6O5r1enW8fTPFvgF0NoCeOoN+ePsqjaBOEvU5x
# hOLKNOhE88BlG//A1h0MoIAkfrjuKwQ6+gCwwH19E8JaRSat3s5o9WleNJv0lQUw
# XvP6snACbJO+ifCpwXbt2MnK9LNpcVReY3745EIMOhJ7ed7YNJDTyagluEZuWF1h
# x1BdyrNiisOC4mTF2ZtdMS5fFRNRLphu3lU1b9X+j058o7ekWZAfQJzaSW4wSDCL
# WmfHrLvsHJ0iOvYmZ3sLYySpxbco5nLtUI1chNAboKPnCyw+REqXRV8M6M3RwOMO
# mySDm0dcifHbxNWE7OdBrveqSZmm/AMz8keTO3EwNCpFAqr8PZ/QpqYqAs0NUAFn
# h+336Bbz1rCk7+AOJ9kTcAC0UEF2tr9fX/w1eNw54CCKG5STd8mms7pbNWzB+wpS
# J4qB5Z4xFLcd3jHuNK2SWr95ZaKqogS5WCJfCVwSXM7xzZzh74yt7DQ6n/l1yKO0
# pZO9d6e6uze8ovDzphKZdas9aalVfnoCcKKa2A7vwCCPY9t+uuGNGCXEZ05ExGvR
# MM5H9f/YsBQMTdXUWVoNL/aTQSR0PYs=
# SIG # End signature block
