//
//  AdPopcornSSPBannerAdapter.m
//  AdPopcornSSP
//
//  Created by 김민석 on 2021/10/21.
//  Copyright © 2021 AdPopcorn. All rights reserved.
//

#import "AdPopcornSSPBannerAdapter.h"
#import "AdPopcornSSPBannerView.h"
#import "AdPopcornSSPAdSize.h"

// Private state for the adapter that also specifies conformance to `ReferenceNetworkInlineAdDelegate`.
@interface AdPopcornSSPBannerAdapter() <APSSPBannerViewDelegate>
@property (nonatomic, strong) AdPopcornSSPBannerView *bannerView;
@end

@implementation AdPopcornSSPBannerAdapter

#pragma mark - Required MPInlineAdAdapter Implementation

// Let the `MPInlineAdAdapter` base class handle the implementation of `delegate` and `localExtras`.
@dynamic delegate;
@dynamic localExtras;

// To opt-out of the MoPub SDK automatic impression and click counting, this method should return `NO`.
// When opted out, it is the network SDK's responsibility to track their own impressions.
- (BOOL)enableAutomaticImpressionAndClickTracking {
    return YES;
}

// Ad load entry point.
- (void)requestAdWithSize:(CGSize)size adapterInfo:(NSDictionary *)info adMarkup:(NSString * _Nullable)adMarkup {
    // `info` contains a combination of MoPubSDK passed in information and
    // network SDK-sepcific information from the MoPub UI. Parse out the information
    // that is needed to create the appropriate network SDK classes.
    // One example is provided below showing logic to determine if the inline ad is a banner
    // or medium rectangle format.
    
    // Determine if the inline ad format is a banner or medium rectangle.
    NSString *appKey = info[@"appKey"];
    NSString *placementId = info[@"placementId"];
    NSString *bannerSize = info[@"bannerSize"];
    if (placementId.length == 0) {
        NSError *error = [NSError noPlacementId];
        [self inlineAdDidFailToLoad:error];
        return;
    }
    
    // Initialize an instance of the network SDK's inline ad.
    _bannerView = [[AdPopcornSSPBannerView alloc] initWithBannerViewSize:SSPBannerViewSize320x50 origin:CGPointMake(0.0f, self.view.frame.size.height - [AdPopcornSSPAdSize adSize:SSPBannerViewSize320x50].height*2) appKey:appKey placementId:placementId viewController:self];
    
    // Load the ad.
    [_bannerView loadRequest];
}

#pragma mark - ReferenceNetworkInlineAdDelegate

/// Ad Clickthrough
- (void)inlineAdClicked {
    // Notify that the clickthrough was tapped and tracked
    [self.delegate inlineAdAdapterDidTrackClick:self];
    
    // Notify that the app will navigate to the clickthrough destination.
    [self.delegate inlineAdAdapterWillLeaveApplication:self];
}

/// Failed banner load.
- (void)inlineAdDidFailToLoad:(NSError *)error {
    // Report the failure to load back to the MoPub SDK.
    [self.delegate inlineAdAdapter:self didFailToLoadAdWithError:error];
}

/// Failed banner show.
- (void)inlineAdDidFailToShow:(NSError *)error {
    // Report the failure to show back to the MoPub SDK.
    [self.delegate inlineAdAdapter:self didFailToLoadAdWithError:error];
}

/// Successful banner load.
- (void)inlineAdDidLoad {
    // Inline ad loading and showing for this Reference Network SDK are seperate.
    // Currently the MoPub SDK combines load and show into a single action.
    
    // This may change in the future, but for now a successful load must
    // trigger a show attempt, and the result of the show will report
    // the final status back to the MoPub SDK.
    [self.inlineAd show];
}

/// Successful banner show.
- (void)inlineAdDidShow:(UIImageView *)creative {
    // Report the success of the show and give back the creative view to
    // the MoPub SDK for rendering onscreen.
    [self.delegate inlineAdAdapter:self didLoadAdWithAdView:creative];
}

/// Impression fired.
- (void)inlineAdImpressionTracked {
    // Notify that the impression was tracked by the network.
    [self.delegate inlineAdAdapterDidTrackImpression:self];
}

@end
