#!/bin/bash
# This script automates the setup and execution of the webcam detection application using standard Python tools.

# Ensure the script runs from its own directory to handle relative paths correctly.
cd "$(dirname "$0")"

# --- 0. Check for Python 3 --- 
if ! command -v python3 &> /dev/null; then
    echo "Error: python3 is not installed or not in PATH. Please install Python 3."
    exit 1
fi

# --- 1. Setup Virtual Camera Device ---
echo "Setting up virtual camera device /dev/video10..."
# Try to load the v4l2loopback module.
sudo modprobe v4l2loopback devices=1 video_nr=10 card_label="VirtualCam" exclusive_caps=1

# Check if the previous command failed.
if [ $? -ne 0 ]; then
    echo "Warning: 'modprobe v4l2loopback' failed. The module might not be installed."
    echo "Attempting to install v4l2loopback-dkms via apt..."
    
    # Update apt package list and install the module non-interactively.
    sudo apt update
    sudo apt install v4l2loopback-dkms -y
    
    # Retry loading the module after the installation attempt.
    echo "Installation attempted. Retrying modprobe..."
    sudo modprobe v4l2loopback devices=1 video_nr=10 card_label="VirtualCam" exclusive_caps=1
    
    # If it still fails, just print a warning and continue.
    if [ $? -ne 0 ]; then
        echo "⚠️ Warning: Failed to load v4l2loopback module even after attempting installation."
        echo "The script will continue, but may rely on the Python script's fallback camera."
    fi
fi

echo "✅ v4l2loopback module is loaded successfully."

# --- 2. Setup Python Virtual Environment ---
# Checks for the .venv directory and creates it if not found.
if [ ! -d ".venv" ]; then
    echo "Virtual environment not found. Creating it with 'python3 -m venv'..."
    # First attempt to create the virtual environment.
    python3 -m venv .venv --system-site-packages

    # Check if venv creation failed (e.g., python3-venv is not installed).
    if [ $? -ne 0 ]; then
        echo "Warning: 'python3 -m venv' failed. Attempting to install 'python3-venv' via apt..."
        sudo apt update
        sudo apt install python3-venv -y

        echo "Installation attempted. Retrying to create virtual environment..."
        # Second attempt to create the venv.
        python3 -m venv .venv --system-site-packages

        # If it still fails, exit. We cannot proceed without a virtual environment.
        if [ $? -ne 0 ]; then
            echo "Error: Failed to create virtual environment even after attempting installation."
            echo "Please check your Python 3 installation and ensure venv can be used."
            exit 1
        fi
    fi
    
    if [ -f "requirements.txt" ]; then
        echo "Installing dependencies from requirements.txt..."
        # Use the pip from the newly created virtual environment to install packages.
        .venv/bin/python -m pip install -r requirements.txt

        # Uninstall numpy and opencv from the venv to force using the system-wide packages
        echo "Uninstalling numpy and opencv-python from venv to use system-installed versions..."
        .venv/bin/python -m pip uninstall numpy opencv-python -y
    else
        echo "Warning: requirements.txt not found. Skipping dependency installation."
    fi
    echo "Setup complete."
fi

# --- 3. Start Camera Pipeline in Background ---
echo "Starting GStreamer camera pipeline in the background..."
gst-launch-1.0 nvarguscamerasrc ! 'video/x-raw(memory:NVMM),width=1920,height=1080,format=NV12,framerate=30/1' ! nvvidconv ! 'video/x-raw, format=I420' ! videoconvert ! 'video/x-raw, format=BGR' ! v4l2sink device=/dev/video10 > gstreamer_pipeline.log 2>&1 &

# Store the Process ID (PID) of the last background command (the GStreamer pipeline).
GST_PID=$!

# --- 4. Setup Cleanup Trap ---
# This defines a command to be executed when the script exits for any reason.
# It ensures the background GStreamer process is stopped cleanly.
trap "echo 'Stopping camera pipeline (PID: $GST_PID)...'; kill $GST_PID" EXIT

# --- 5. Wait for Pipeline to Initialize ---
# Give the GStreamer pipeline a moment to start up to avoid a race condition.
echo "Waiting 3 seconds for camera pipeline to initialize..."
sleep 3

# --- 6. Run Main Application ---
# This runs the main Python script using the interpreter from the virtual environment.
echo "Starting main detection application..."
.venv/bin/python webcam_yolo.py

# --- 7. Automatic Cleanup ---
# The 'trap' command set earlier will now execute automatically, killing the GStreamer process.
echo "Application finished. Cleanup will be handled automatically."
